/* This is a script that lists the day's callers.
 *
 * Author: Eric Oulashin (AKA Nightfox)
 * BBS: Digital Distortion
 * BBS address: digdist.bbsindex.com
 *
 * Date       User              Description
 * 2009-04-30 Eric Oulashin     Created
 * 2009-05-01 Eric Oulashin     Continued work
 * 2009-05-02 Eric Oulashin     Continued work - Used substring to make
 *                              sure field values don't exceed the field
 *                              widths.  Updated to read logon.lst in the
 *                              data directory rather than going through
 *                              the user records and matching their last
 *                              logon date.
 * 2009-05-03 Eric Oulashin     Refactored the script a bit.  Added a
 *                              boolean, pauseAtEnd, which will control
 *                              whether or not the script will pause at
 *                              the end, and set up argv[0] to change
 *                              that too.  Also, added a Synchronet
 *                              version check.
 * 2009-05-04 Eric Oulashin     Changed the User variable's name from user
 *                              to theUser to avoid stomping on the user
 *                              object declared in sbbsdefs.js.  Also,
 *                              updated to read through the system user
 *                              records if unable to read the logon.lst file.
 *                              Either way, the caller information is now
 *                              placed in an array, and the last X callers
 *                              are displayed at the end.  Also, made use
 *                              of the newly-added variable DIGDIST_INCLUDE_SYSOP
 *                              in DigitalDistortionDefs.js.
 * 2009-05-09 Eric Oulashin     Updated to only add the caller information to
 *                              the array if the user's handle & logon time
 *                              read from logon.lst are not blank.  This fixed
 *                              a bug that was causing the script to occasionally
 *                              show a single line in the logon list for the day's
 *                              first caller.
 * 2009-05-10 Eric Oulashin     Removed the "donePrompt" color from the colors
 *                              array, because it wasn't being used.
 * 2009-05-13 Eric Oulashin     Updated to be free of DigitalDistortionDefs.js.
 *                              Added VERSION and VER_DATE, although they aren't
 *                              displayed to the user.
 * 2009-06-16 Eric Oulashin     Version 1.02
 *                              - Put the caller listing code into a class so that
 *                                the caller list can either be executed in this
 *                                script (default), or this script can be loaded
 *                                into another script, and the user may use the
 *                                class as desired.
 *                              - Added parameters to the user listing functions
 *                                to specify whether to return the caller list as
 *                                as a string rather than outputting it to the
 *                                screen.
 * 2009-07-18 Eric Oulashin     Version 1.03
 *                              Updated to add the user number to the CallerInfo
 *                              object.  Also, added getHTML(), which returns
 *                              the day's callers list as an HTML table.  Also,
 *                              when parsing data/logon.lst, this script now adds
 *                              1 to the call #, because it seems to be 0-based
 *                              in that file.  Also, made the node # and # calls
 *                              right-justified, since they are numeric fields.
 * 2009-07-31 Eric Oulashin     Updated centerText() to get the width of the
 *                              text after using strip_ctrl() so that it
 *                              doesn't count control characters.
 * 2009-08-19 Eric Oulashin     Version 1.04
 *                              Updated getHTML() to center the column header
 *                              text in the table.
 *                              Added the feature of reading configuration
 *                              settings from a configuration file.
 *                              Added an option for whether or not to display
 *                              QWK account logins.
 *                              Removed the section for looping through all
 *                              users in case logon.lst couldn't be opened -
 *                              That looped went from 1 to system.stats.total_users,
 *                              but the error with that is that user numbers
 *                              aren't necessarily sequential.
 * 2011-11-24 Eric Oulashin     Version 1.05
 *                              Updated to report the # of calls more correctly.
 *                              Thanks to Slinky for reporting this.
 * 2018-08-03 Eric Oulashin     Version 1.06
 *                              Updated to delete instances of User objects that
 *                              are created, due to an optimization in Synchronet
 *                              3.17 that leaves user.dat open
 * 2022-06-07 Eric Oulashin     Version 1.07
 *                              Now skips lines from the login file that are not strings, to
 *                              avoid errors with fileLine being undefined.
 * 2023-03-20 Eric Oulashin     Version 1.08
 *                              Defined the high-ASCII characters using hex codes. No
 *                              functional change.
 */

/* These are the command-line arguments:
 * 0: Whether or not to pause at the end of the listing (true/false)
 * 1: The maximum number of callers to list
 * 2: Whether or not to execute the caller list (true/false).  This
 *    dfaults to true.  You would want to pass false if you are including
 *    this script in another JavaScript file and wish to execute it in your
 *    own way.
 */

"use strict";

if (typeof(require) === "function")
	require("sbbsdefs.js", "K_NOCRLF");
else
	load("sbbsdefs.js");


// This script requires Synchronet version 3.10 or higher.
// Exit if the Synchronet version is below the minimum.
if (system.version_num < 31000)
{
	var message = "\x01n\x01h\x01y\x01i* Warning:\x01n\x01h\x01w Digital Distortion List Today's Callers "
	             + "requires version \x01g3.10\x01w or\r\n"
	             + "higher of Synchronet.  This BBS is using version \x01g" + system.version
	             + "\x01w.  Please notify the sysop.";
	console.crlf();
	console.print(message);
	console.crlf();
	console.pause();
	exit();
}

// Version and version date.
// These are declared with "var" rather than "const" to avoid
// an error about re-definitions.
var VERSION = "1.08";
var VER_DATE = "2023-03-20";

// Single-line box drawing/border characters.
// These are declared with "var" rather than "const" to avoid
// an error about re-definitions.
var UPPER_LEFT_SINGLE = "\xDA";
var HORIZONTAL_SINGLE = "\xC4";
var UPPER_RIGHT_SINGLE = "\xBF";
var VERTICAL_SINGLE = "\xB3";
var LOWER_LEFT_SINGLE = "\xC0";
var LOWER_RIGHT_SINGLE = "\xD9";
var T_SINGLE = "\xC2";
var LEFT_T_SINGLE = "\xC3";
var RIGHT_T_SINGLE = "\xB4";
var BOTTOM_T_SINGLE = "\xC1";
var CROSS_SINGLE = "\xC5";
// Double-line box drawing/border characters.
var UPPER_LEFT_DOUBLE = "\xC9";
var HORIZONTAL_DOUBLE = "\xCD";
var UPPER_RIGHT_DOUBLE = "\xBB";
var VERTICAL_DOUBLE = "\xBA";
var LOWER_LEFT_DOUBLE = "\xC8";
var LOWER_RIGHT_DOUBLE = "\xBC";
var T_DOUBLE = "\xCB";
var LEFT_T_DOUBLE = "\xCC";
var RIGHT_T_DOUBLE = "\xB9";
var BOTTOM_T_DOUBLE = "\xCA";
var CROSS_DOUBLE = "\xCE";

///////////////////////////
// Main script execution //
///////////////////////////

// The 3rd program argument specifies whether or not to execute the script code.
var executeScriptCode = true;
if ((typeof(argv[2]) != "undefined") && (argv[2] != null))
   executeScriptCode = argv[2];

if (executeScriptCode)
{
	var callerLister = new DDTodaysCallerLister();

	// The first program argument can change pauseAtEnd.
	if ((typeof(argv[0]) != "undefined") && (argv[0] != null))
		callerLister.pauseAtEnd = argv[0];
	// The second program argument can change MAX_CALLERS.
	if ((typeof(argv[1]) != "undefined") && (argv[1] != null))
		callerLister.MAX_CALLERS = argv[1];

	callerLister.listTodaysCallers(false);
}

// End of script execution


//////////////////////////////////////////////////////////////////////////////////
// Class & functions

// This is the constructor for the CallerInfo class, which contains
// information about a caller.
function CallerInfo()
{
	this.nodeNumber = "1";
	this.handle = "";
	this.location = "";
	this.logonTime = "0";
	this.connectionType = "";
	this.numCallsToday = 0;
	this.userNumber = 0;
}

// This is the constructor for the DDTodaysCallerLister class, which
// performs a listing of the day's callers.
function DDTodaysCallerLister()
{
	// Colors
	this.colors = {
		veryTopHeader:  "\x01n\x01h\x01w",
		border:  "\x01n\x01c",
		colLabelText:  "\x01n\x014\x01h\x01w",
		timeOn:  "\x01h\x01c",
		userName:  "\x01h\x01y",
		location:  "\x01h\x01m",
		nodeNumber:  "\x01h\x01r",
		connectionType:  "\x01h\x01b",
		numCalls:  "\x01h\x01g"
	};

	// pauseAtEnd controls whether or not to pause at the end of the listing.
	this.pauseAtEnd = true;
	// The maximum number of callers to display
	this.MAX_CALLERS = 6;
	// Whether or not to include the sysop in the list.
	this.includeSysop = true;
	// Whether or not to output the list as HTML (using html_encode()).
	this.doHTML = false;
	// Whether or not to display QWK accounts
	this.displayQWKAccts = false;

	// Column inner lengths
	this.TIME_ON_LEN = 7;
	this.USER_NAME_LEN = 18;
	this.LOCATION_LEN = 25;
	this.NODE_NUM_LEN = 5;
	this.CONNECTION_TYPE_LEN = 10;
	this.NUM_CALLS_LEN = 6;

	// This is the message that is displayed if the user is the first caller of the day.
	this.firstCallerMessage = this.colors["veryTopHeader"] + "You are the first caller of the day!";

	// callerInfoArray is an array of CallerInfo objects containing
	// the day's caller information.
	this.callerInfoArray = new Array();

	// Member methods
	this.populateCallerInfoArray = DDTodaysCallerLister_PopulateCallerInfoArray;
	this.listTodaysCallers = DDTodaysCallerLister_ListTodaysCallers;
	this.writeHeader = DDTodaysCallerLister_WriteHeader;
	this.writeEndBorder = DDTodaysCallerLister_WriteEndBorder;
	this.getHTML = DDTodaysCallerLister_GetHTML;
	this.readConfigFile = DDTodaysCallerLister_ReadConfigFile;

	// Read the configuration file to set the settings from there.
	this.readConfigFile();
}
// For the DDTodaysCallerLister class: Populates the array of callers.
function DDTodaysCallerLister_PopulateCallerInfoArray()
{
	// Clear the caller info array
	this.callerInfoArray = [];

	// If there weren't any logins today, then just return now.
	if (system.stats.logons_today == 0)
		return;

	// Try to open logon.lst in the data dir.
	var logonLstFilename = file_getcase(system.data_dir + "logon.lst");
	if (logonLstFilename == undefined)
		return;

	// Open logon.lst and read the caller information from it.
	var loginFileOpened = false;
	var loginFile = new File(logonLstFilename);
	if (loginFile.open("r", true))
	{
		// Read the lines from the file and insert the caller information into
		// this.callerInfoArray.
		var fileLine = "";
		var userHandle = ""; // For testing for valid data
		var logonTime = "";  // For testing for valid data
		var callerInfo = null;
		while (!loginFile.eof)
		{
			fileLine = loginFile.readln(512);

			// Skip blank lines
			if (typeof(fileLine) !== "string") // Should skip undefined/null lines
				continue;
			if (fileLine == "")
				continue;

			fileLine = strip_ctrl(fileLine);

			/* Fields
			------
			0-2: Node #
			3-9: Call number (or previous number of calls)
			10-35: User handle
			36-60: User location
			61-65: Login time
			67-74: Connection type
			76-80: The user's number of calls today
			*/

			userHandle = trimSpaces(fileLine.substring(10, 36), false, false, true);

			// If we're set up to not include the sysop, and this is the sysop,
			// then skip this user.
			var userNum = system.matchuser(userHandle, true);
			if (!this.includeSysop && (userNum == 1))
				continue;

			// If we are to skip QWK accounts, then check to see if this is a
			// QWK account, and if so, then skip it.
			if (!this.displayQWKAccts)
			{
				var theUser = new User(userNum);
				var userIsQWK = theUser.compare_ars("REST Q");
				theUser = undefined; // Destructs the object now, rather than with 'delete'
				if (userIsQWK)
					continue;
			}

			// Only add the caller info if the user's handle & logon time are
			// not blank.
			var userLogonTime = trimSpaces(fileLine.substring(61, 66), false, false, true);
			if ((userHandle != "") && (userLogonTime != ""))
			{
				callerInfo = new CallerInfo();
				callerInfo.nodeNumber = trimSpaces(fileLine.substring(0, 2), false, false, true);
				callerInfo.handle = userHandle;
				callerInfo.location = trimSpaces(fileLine.substring(36, 61), false, false, true);
				callerInfo.logonTime = userLogonTime;
				callerInfo.connectionType = trimSpaces(fileLine.substring(67, 75), false, false, true);
				//callerInfo.numCallsToday = (+(trimSpaces(fileLine.substring(76, 81), false, false, true)) + 1).toString();
				callerInfo.numCallsToday = (+(trimSpaces(fileLine.substring(76, 81), false, false, true))).toString();
				callerInfo.userNumber = system.matchuser(callerInfo.handle);
				this.callerInfoArray.push(callerInfo);
			}
		}
		loginFile.close();
	}
}
// For the DDTodaysCallerLister class: Lists the day's callers.
//
// Parameters:
//  pReturnInString: Boolean - Whether or not to return the caller listing
//                   as a string rather than outputting it to the scren.
//                   This is options.
function DDTodaysCallerLister_ListTodaysCallers(pReturnInString)
{
	var returnInString = false;
	if (pReturnInString != null)
		returnInString = pReturnInString;

	if (!returnInString)
		console.clear();

	var returnStr = "";

	// Populate the caller info array
	this.populateCallerInfoArray();

	// If this.callerInfoArray has any caller information, then display it.
	if (this.callerInfoArray.length > 0)
	{
		var lineText = "";

		// The following line which sets console.line_counter to 0 is a
		// kludge to disable Synchronet's automatic pausing after a
		// screenful of text.
		if (!returnInString)
			console.line_counter = 0;

		// Write the caller list header
		returnStr = this.writeHeader(returnInString);

		// We want to display the last MAX_CALLERS number of callers.  First,
		// calculate the starting array index based on MAX_CALLERS.  If
		// MAX_CALLERS is 0 or negative, then display all the callers in the
		// array.
		var startIndex = 0;
		if ((this.MAX_CALLERS > 0) && (this.callerInfoArray.length > this.MAX_CALLERS))
			startIndex = this.callerInfoArray.length - this.MAX_CALLERS;
		// Display the callers.
		for (var i = startIndex; i < this.callerInfoArray.length; ++i)
		{
			lineText = this.colors.border + VERTICAL_SINGLE;
			lineText += this.colors.timeOn + centerText(this.callerInfoArray[i].logonTime, this.TIME_ON_LEN);
			lineText += this.colors.border + VERTICAL_SINGLE;
			lineText += this.colors.userName + centerText(this.callerInfoArray[i].handle.substring(0, this.USER_NAME_LEN), this.USER_NAME_LEN);
			lineText += this.colors.border + VERTICAL_SINGLE;
			lineText += this.colors.location + centerText(this.callerInfoArray[i].location.substring(0, this.LOCATION_LEN), this.LOCATION_LEN);
			lineText += this.colors.border + VERTICAL_SINGLE;
			lineText += this.colors.nodeNumber + format("%" + this.NODE_NUM_LEN + "d", this.callerInfoArray[i].nodeNumber);
			lineText += this.colors.border + VERTICAL_SINGLE;
			lineText += this.colors.connectionType;
			lineText += centerText(this.callerInfoArray[i].connectionType.substring(0, this.CONNECTION_TYPE_LEN), this.CONNECTION_TYPE_LEN);
			lineText += this.colors.border + VERTICAL_SINGLE;
			lineText += this.colors.numCalls;
			lineText += centerText(format("%" + this.NUM_CALLS_LEN + "d", this.callerInfoArray[i].numCallsToday), this.NUM_CALLS_LEN);
			lineText += this.colors.border + VERTICAL_SINGLE;
			if (returnInString)
				returnStr += lineText + "\r\n";
			else
				console.center(lineText);
		}

		// Write the end border for the table
		returnStr += this.writeEndBorder(returnInString);
	}
	else
	{
		// There are no elements in this.callerInfoArray, so tell the user that they
		// are the first caller.
		if (returnInString)
			returnStr = this.firstCallerMessage;
		else
			console.center(this.firstCallerMessage);
		}

	// We're done.  If pauseAtEnd is true, then display the system's pause/"Hit a key" prompt.
	if (this.pauseAtEnd && !returnInString)
		console.print("\x01n\x01p");

	return returnStr;
}
// For the DDTodaysCallerLister class: Writes the header for the user listing.
//
// Parameters:
//  pReturnInString: Boolean - Whether or not to return the caller listing
//                   as a string rather than outputting it to the scren.
//                   This is options.
function DDTodaysCallerLister_WriteHeader(pReturnInString)
{
	var returnInString = false;
	if (pReturnInString != null)
	returnInString = pReturnInString;

	var returnStr = "";

	// Write the header
	if (returnInString)
	{
		var tableWidth = this.TIME_ON_LEN + this.USER_NAME_LEN + this.LOCATION_LEN
		               + this.NODE_NUM_LEN + this.CONNECTION_TYPE_LEN + this.NUM_CALLS_LEN + 7;
		returnStr = this.colors["veryTopHeader"] + centerText("Last several callers", tableWidth) + "\r\n";
	}
	else
	{
		console.print(this.colors["veryTopHeader"]);
		console.center("Last several callers");
	}
	// Upper border lines
	// "Time on" section
	lineText = this.colors["border"] + UPPER_LEFT_SINGLE;
	for (var i = 0; i < this.TIME_ON_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += T_SINGLE;
	// User name section
	for (var i = 0; i < this.USER_NAME_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += T_SINGLE;
	// Location section
	for (var i = 0; i < this.LOCATION_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += T_SINGLE;
	// Node number section
	for (var i = 0; i < this.NODE_NUM_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += T_SINGLE;
	// Connection type section
	for (var i = 0; i < this.CONNECTION_TYPE_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += T_SINGLE;
	// # Calls section
	for (var i = 0; i < this.NUM_CALLS_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += UPPER_RIGHT_SINGLE;
	if (returnInString)
		returnStr += lineText + "\r\n";
	else
		console.center(lineText + "\r\n");

	// Column labels
	lineText = this.colors["border"] + VERTICAL_SINGLE + this.colors["colLabelText"]
	         + centerText("Time On", this.TIME_ON_LEN) + this.colors["border"] + VERTICAL_SINGLE
	         + this.colors["colLabelText"] + centerText("User Name", this.USER_NAME_LEN) + this.colors["border"]
	         + VERTICAL_SINGLE + this.colors["colLabelText"]
	         + centerText("Location", this.LOCATION_LEN) + this.colors["border"] + VERTICAL_SINGLE
	         + this.colors["colLabelText"]
	         + centerText("Node#", this.NODE_NUM_LEN)  + this.colors["border"] + VERTICAL_SINGLE
	         + this.colors["colLabelText"] + centerText("Connection", this.CONNECTION_TYPE_LEN)
	         + this.colors["border"] + VERTICAL_SINGLE + this.colors["colLabelText"]
	         + centerText("#Calls", this.NUM_CALLS_LEN) + this.colors["border"]
	         + VERTICAL_SINGLE;
	if (returnInString)
		returnStr += lineText + "\r\n";
	else
		console.center(lineText + "\r\n");

	// Lower border lines for header
	// "Time on" section
	lineText = this.colors["border"] + LEFT_T_SINGLE;
	for (var i = 0; i < this.TIME_ON_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += CROSS_SINGLE;
	// User name section
	for (var i = 0; i < this.USER_NAME_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += CROSS_SINGLE;
	// Location section
	for (var i = 0; i < this.LOCATION_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += CROSS_SINGLE;
	// Node number section
	for (var i = 0; i < this.NODE_NUM_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += CROSS_SINGLE;
	// Connection type section
	for (var i = 0; i < this.CONNECTION_TYPE_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += CROSS_SINGLE;
	// # Calls section
	for (var i = 0; i < this.NUM_CALLS_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += RIGHT_T_SINGLE;
	if (returnInString)
		returnStr += lineText + "\r\n";
	else
		console.center(lineText);

	return returnStr;
}
// For the DDTodaysCallerLister class: Writes the end border.
//
// Parameters:
//  pReturnInString: Boolean - Whether or not to return the caller listing
//                   as a string rather than outputting it to the scren.
//                   This is options.
function DDTodaysCallerLister_WriteEndBorder(pReturnInString)
{
	var returnInString = false;
	if (pReturnInString != null)
		returnInString = pReturnInString;

	var returnStr = "";

	// "Time on" section
	var lineText = this.colors["border"] + LOWER_LEFT_SINGLE;
	for (var i = 0; i < this.TIME_ON_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += BOTTOM_T_SINGLE;
	// User name section
	for (var i = 0; i < this.USER_NAME_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += BOTTOM_T_SINGLE;
	// Location section
	for (var i = 0; i < this.LOCATION_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += BOTTOM_T_SINGLE;
	// Node number section
	for (var i = 0; i < this.NODE_NUM_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += BOTTOM_T_SINGLE;
	// Connection type section
	for (var i = 0; i < this.CONNECTION_TYPE_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += BOTTOM_T_SINGLE;
	// # Calls section
	for (var i = 0; i < this.NUM_CALLS_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += LOWER_RIGHT_SINGLE;
	if (returnInString)
		returnStr = lineText;
	else
		console.center(lineText);

	return returnStr;
}
// For the DDTodaysCallerLister class: Writes the end border.
//
// Parameters:
//  pTableClass: The class to use for the table.
function DDTodaysCallerLister_GetHTML(pTableClass)
{
	var callerListHTML = "";

	// Populate the caller info array, and if there were callers today,
	// build an HTML table with the callers; otherwise, set a message
	// saying there have been no callers today.
	this.populateCallerInfoArray();
	if (this.callerInfoArray.length > 0)
	{
		callerListHTML = "<table ";
		if ((pTableClass != null) && (typeof(pTableClass) != "undefined") && (pTableClass.length > 0))
			callerListHTML += "class='" + pTableClass + "' ";
		callerListHTML += "cellspacing='1' cellpadding='2' summary='Last Several Callers'>\n"
		               + "<tr nowrap style='color: #ffffff; background-color: #0000bb; text-align: center;'>"
		               + "<td>Time On</td><td>User Name</td><td>Location</td>"
		               + "<td>Node #</td><td>Connection Type</td><td># Calls</td></tr>\n";
		for (var i = 0; i < this.callerInfoArray.length; ++i)
		{
			callerListHTML += "<tr nowrap align='center' style='background-color: #000000;'>"
			               + "<td style='color: #00ffff;'>" + this.callerInfoArray[i].logonTime + "</td>"
			               // Show the user's handle with a link to their profile.
			               + "<td><a href='/members/viewprofile.ssjs?showuser="
			               + this.callerInfoArray[i].userNumber + "' style='color: #ffff00;'>"
			               + this.callerInfoArray[i].handle + "</a></td>"
			               + "<td style='color: #ff00ff;'>" + this.callerInfoArray[i].location + "</td>"
			               + "<td style='color: #ff0000; text-align: right;'>" + this.callerInfoArray[i].nodeNumber + "</td>"
			               + "<td style='color: #0000ff;'>" + this.callerInfoArray[i].connectionType + "</td>"
			               + "<td style='color: #00ff00; text-align: right;'>" + this.callerInfoArray[i].numCallsToday + "</td>"
			               + "</tr>\n";
		}
		callerListHTML += "</table>\n";
	}
	else
		callerListHTML = "<h3>Nobody has called yet today.</h3>";

	return callerListHTML;
}
// For the DDTodaysCallerLister class: Reads the configuration file and sets
//  the various options.
function DDTodaysCallerLister_ReadConfigFile()
{
	// Determine the script's startup directory.
	// This code is a trick that was created by Deuce, suggested by Rob Swindell
	// as a way to detect which directory the script was executed in.  I've
	// shortened the code a little.
	var startup_path = '.';
	try { throw dig.dist(dist); } catch(e) { startup_path = e.fileName; }
	startup_path = backslash(startup_path.replace(/[\/\\][^\/\\]*$/,''));

	// Open the configuration file
	var cfgFile = new File(startup_path + "DigitalDistortionListTodaysCallers.cfg");
	if (cfgFile.open("r"))
	{
		var settingsMode = "behavior";
		var fileLine = null;     // A line read from the file
		var equalsPos = 0;       // Position of a = in the line
		var commentPos = 0;      // Position of the start of a comment
		var setting = null;      // A setting name (string)
		var settingUpper = null; // Upper-case setting name
		var value = null;        // A value for a setting (string)
		while (!cfgFile.eof)
		{
			// Read the next line from the config file.
			fileLine = cfgFile.readln(2048);

			// fileLine should be a string, but I've seen some cases
			// where it isn't, so check its type.
			if (typeof(fileLine) != "string")
				continue;

			// If the line starts with with a semicolon (the comment
			// character) or is blank, then skip it.
			if ((fileLine.substr(0, 1) == ";") || (fileLine.length == 0))
				continue;

			// If in the "behavior" section, then set the behavior-related variables.
			if (fileLine.toUpperCase() == "[BEHAVIOR]")
			{
				settingsMode = "behavior";
				continue;
			}
			else if (fileLine.toUpperCase() == "[COLORS]")
			{
				settingsMode = "colors";
				continue;
			}

			// If the line has a semicolon anywhere in it, then remove
			// everything from the semicolon onward.
			commentPos = fileLine.indexOf(";");
			if (commentPos > -1)
				fileLine = fileLine.substr(0, commentPos);

			// Look for an equals sign, and if found, separate the line
			// into the setting name (before the =) and the value (after the
			// equals sign).
			equalsPos = fileLine.indexOf("=");
			if (equalsPos > 0)
			{
				// Read the setting & value, and trim leading & trailing spaces.
				setting = trimSpaces(fileLine.substr(0, equalsPos), true, false, true);
				settingUpper = setting.toUpperCase();
				value = trimSpaces(fileLine.substr(equalsPos+1), true, false, true).toUpperCase();

				if (settingsMode == "behavior")
				{
					// Set the appropriate setting.
					if (settingUpper == "INCLUDESYSOP")
						this.includeSysop = (value == "TRUE");
					else if (settingUpper == "DISPLAYQWKACCTS")
						this.displayQWKAccts = (value == "TRUE");
					else if (settingUpper == "PAUSEATEND")
						this.pauseAtEnd = (value == "TRUE");
				}
				else if (settingsMode == "colors")
				this.colors[setting] = value;
			}
		}

		cfgFile.close();
	}
}


// The following variable is used for a line of text throughout this script.
var lineText = "";




///////////////////////////////////////////////////////////////////////////////////
// Functions

// Removes multiple, leading, and/or trailing spaces
// The search & replace regular expressions used in this
// function came from the following URL:
//  http://qodo.co.uk/blog/javascript-trim-leading-and-trailing-spaces
//
// Parameters:
//  pString: The string to trim
//  pLeading: Whether or not to trim leading spaces (optional, defaults to true)
//  pMultiple: Whether or not to trim multiple spaces (optional, defaults to true)
//  pTrailing: Whether or not to trim trailing spaces (optional, defaults to true)
function trimSpaces(pString, pLeading, pMultiple, pTrailing)
{
	var leading = true;
	var multiple = true;
	var trailing = true;
	if(typeof(pLeading) != "undefined")
		leading = pLeading;
	if(typeof(pMultiple) != "undefined")
		multiple = pMultiple;
	if(typeof(pTrailing) != "undefined")
		trailing = pTrailing;
		
	// To remove both leading & trailing spaces:
	//pString = pString.replace(/(^\s*)|(\s*$)/gi,"");

	if (leading)
		pString = pString.replace(/(^\s*)/gi,"");
	if (multiple)
		pString = pString.replace(/[ ]{2,}/gi," ");
	if (trailing)
		pString = pString.replace(/(\s*$)/gi,"");

	return pString;
}

// Centers some text within a specified field length.
//
// Parameters:
//  pText: The text to center
//  pFieldLen: The field length
function centerText(pText, pFieldLen)
{
	var centeredText = "";

	var textLen = strip_ctrl(pText).length;
	// If pFieldLen is less than the text length, then truncate the string.
	if (pFieldLen < textLen)
		centeredText = pText.substring(0, pFieldLen);
	else
	{
		// pFieldLen is at least equal to the text length, so we can
		// center the text.
		// Calculate the number of spaces needed to center the text.
		var numSpaces = pFieldLen - textLen;
		if (numSpaces > 0)
		{
			var rightSpaces = (numSpaces/2).toFixed(0);
			var leftSpaces = numSpaces - rightSpaces;
			// Build centeredText
			for (var i = 0; i < leftSpaces; ++i)
				centeredText += " ";
			centeredText += pText;
			for (var i = 0; i < rightSpaces; ++i)
				centeredText += " ";
		}
		else
		{
			// pFieldLength is the same length as the text.
			centeredText = pText;
		}
	}

	return centeredText;
}