// Displays holiday screens. Meant to be run via logon event.
// Version 1.01

"use strict";

require("http.js", "HTTPRequest");

// Read the configuration
var config = readConfig();

// Display the current holiday screen, if applicable.  Display US holiday
// screens only if the user is located in the US or is on the same network
// as the BBS machine.
var gCurDate = time();
var gCurMonthNum = +(strftime("%m", gCurDate));
var gCurDayofMonthNum = +(strftime("%d", gCurDate));
var gUserCountryInfo = getUserCountry();

// Go through the configuration entries and display a holiday screen if
// it's a date to display one
for (var i = 0; i < config.entries.length; ++i)
{
	// If the country code is specified, then only display it if the user's country
	// matches
	var canDisplay = true;
	if (config.entries[i].countryCode != "")
		canDisplay = (gUserCountryInfo.userCountry == config.entries[i].countryCode || config.BBSCountry == config.entries[i].countryCode);
	if (!canDisplay)
		continue;

	// If the current month (and probably day too) matches, then display the screen
	var dayMatch = true;
	if (config.entries[i].day > 0)
		dayMatch = (gCurDayofMonthNum == config.entries[i].day);
	if (gCurMonthNum == config.entries[i].month && dayMatch)
	{
		console.clear();
		if (typeof(config.entries[i].screen) === "string")
			bbs.menu(config.entries[i].screen);
		else if (Array.isArray(config.entries[i].screen) && config.entries[i].screen.length > 0)
		{
			var arrayIdx = random(config.entries[i].screen.length);
			bbs.menu(config.entries[i].screen[arrayIdx]);
		}
	}
}

// Gets an abbreviation for the user's country
function getUserCountry()
{
	var retObj = {
		userIPLocalnet: false,
		userCountry: ""
	};

	if ((client.ip_address == "127.0.0.1") || (client.ip_address.indexOf("192.168.") == 0))
	{
		retObj.userIPLocalnet = true;
		return retObj;
	}

	try
	{
		var contents = new HTTPRequest().Get("http://www.geoplugin.net/json.gp?ip=" + client.ip_address);
		if (contents != "")
		{
			try
			{
				var obj = JSON.parse(contents);
				retObj.userCountry = obj.geoplugin_countryCode;
			}
			catch (error)
			{
				var errorMsg = "!ERROR Unable to look up user's country code based on their IP address (" + client.ip_address + "): " + error;
				log(LOG_ERR, errorMsg);
			}
		}
	}
	catch (error)
	{
		log(LOG_ERR, error);
	}

	return retObj;
}

// Reads the settings/configuration
function readConfig()
{
	// Options from modopts.ini
	var iniObj = load({}, "modopts.js", "dd_holiday_screens");
	if (!iniObj)
		iniObj = {};

	var config = {
		BBSCountry: "",
		entries: []
	};

	for (var prop in iniObj)
	{
		// BBS country
		if (prop == "BBSCountry")
			config.BBSCountry = iniObj[prop];
		else
		{
			// #-#_??: Month-day_CountryCode
			var results = prop.match(/^([0-9]+)-([0-9]+)_([A-Za-z]+)$/);
			if (Array.isArray(results) && results.length >= 4)
			{
				var cfgEntry = getDefaultConfigEntry();
				cfgEntry.month = parseInt(results[1]);
				cfgEntry.day = parseInt(results[2]);
				cfgEntry.countryCode = results[3];
				cfgEntry.screen = parseScreenFilename(iniObj[prop]);
				if (!isNaN(cfgEntry.month) && !isNaN(cfgEntry.day) && itemIsStrOrArrayOfStrsAndNotBlank(cfgEntry.screen))
					config.entries.push(cfgEntry);
			}
			else
			{
				// #-#: Month-day
				results = prop.match(/^([0-9]+)-([0-9]+)$/);
				if (Array.isArray(results) && results.length >= 3)
				{
					cfgEntry = getDefaultConfigEntry();
					cfgEntry.month = parseInt(results[1]);
					cfgEntry.day = parseInt(results[2]);
					cfgEntry.screen = parseScreenFilename(iniObj[prop]);
					if (!isNaN(cfgEntry.month) && !isNaN(cfgEntry.day) && itemIsStrOrArrayOfStrsAndNotBlank(cfgEntry.screen))
						config.entries.push(cfgEntry);
				}
				else
				{
					// #-dayName#_CountryCode
					results = prop.match(/^([0-9]+)-([A-Za-z]+)([0-9]+)_([A-Za-z]+)$/);
					if (Array.isArray(results) && results.length >= 5)
					{
						var dayOfWeekNum = dayOfWeekNameToNum(results[2]);
						if (dayOfWeekNum > -1)
						{
							var cfgEntry = getDefaultConfigEntry();
							cfgEntry.month = parseInt(results[1]);
							var ordinalNum = parseInt(results[3]);
							if (!isNaN(cfgEntry.month) && !isNaN(ordinalNum))
							{
								// Figure out cfgEntry.day. If the day of the month is
								// valid, then add the date entry
								cfgEntry.day = getNthWeekdayDayofMonthForCurrentYear(cfgEntry.month, dayOfWeekNum, ordinalNum);
								if (cfgEntry.day > 0)
								{
									cfgEntry.countryCode = results[4];
									cfgEntry.screen = parseScreenFilename(iniObj[prop]);
									if (itemIsStrOrArrayOfStrsAndNotBlank(cfgEntry.screen))
										config.entries.push(cfgEntry);
								}
							}
						}
					}
					else
					{
						// #-dayName#
						results = prop.match(/^([0-9]+)-([A-Za-z]+)([0-9]+)$/);
						if (Array.isArray(results) && results.length >= 4)
						{
							var dayOfWeekNum = dayOfWeekNameToNum(results[2]);
							if (dayOfWeekNum > -1)
							{
								var cfgEntry = getDefaultConfigEntry();
								cfgEntry.month = parseInt(results[1]);
								var ordinalNum = parseInt(results[3]);
								if (!isNaN(cfgEntry.month) && !isNaN(ordinalNum))
								{
									// Figure out cfgEntry.day. If the day of the month is
									// valid, then add the date entry
									cfgEntry.day = getNthWeekdayDayofMonthForCurrentYear(cfgEntry.month, dayOfWeekNum, ordinalNum);
									if (cfgEntry.day > 0)
									{
										cfgEntry.screen = parseScreenFilename(iniObj[prop]);
										if (itemIsStrOrArrayOfStrsAndNotBlank(cfgEntry.screen))
											config.entries.push(cfgEntry);
									}
								}
							}
						}
						else
						{
							// ##_rand#_<CountryCode>
							results = prop.match(/^([0-9]+)_rand.*_([A-Za-z]+)$/);
							if (Array.isArray(results) && results.length >= 3)
							{
								var cfgEntry = getDefaultConfigEntry();
								cfgEntry.month = parseInt(results[1]);
								if (!isNaN(cfgEntry.month))
								{
									cfgEntry.countryCode = results[2];
									cfgEntry.screen = parseScreenFilename(iniObj[prop]);
									if (itemIsStrOrArrayOfStrsAndNotBlank(cfgEntry.screen))
									{
										// See if there's already an entry in the array for this
										// month, and of so, update its screen property. Otherwise,
										// add the entry to config.entries.
										var foundEntry = false;
										for (var i = 0; i < config.entries.length && !foundEntry; ++i)
										{
											if (config.entries[i].month == cfgEntry.month && config.entries[i].day == 0 && config.entries[i].countryCode == cfgEntry.countryCode)
											{
												if (Array.isArray(config.entries[i].screen))
												{
													if (Array.isArray(cfgEntry.screen))
														config.entries[i].screen = config.entries[i].screen.concat(cfgEntry.screen);
													else
														config.entries[i].screen.push(cfgEntry.screen);
												}
												else
												{
													var tmpArray = [];
													tmpArray.push(config.entries[i].screen)
													if (Array.isArray(cfgEntry.screen))
														tmpArray = tmpArray.concat(cfgEntry.screen);
													else
														tmpArray.push(cfgEntry.screen);
													config.entries[i].screen = tmpArray;
												}
												foundEntry = true;
											}
										}
										if (!foundEntry)
											config.entries.push(cfgEntry);
									}
								}
							}
							else
							{
								// ##_rand#
								results = prop.match(/^([0-9]+)_rand.*$/);
								if (Array.isArray(results) && results.length >= 2)
								{
									var cfgEntry = getDefaultConfigEntry();
									cfgEntry.month = parseInt(results[1]);
									if (!isNaN(cfgEntry.month))
									{
										cfgEntry.screen = parseScreenFilename(iniObj[prop]);
										if (itemIsStrOrArrayOfStrsAndNotBlank(cfgEntry.screen))
										{
											// See if there's already an entry in the array for this
											// month, and of so, update its screen property. Otherwise,
											// add the entry to config.entries.
											var foundEntry = false;
											for (var i = 0; i < config.entries.length && !foundEntry; ++i)
											{
												if (config.entries[i].month == cfgEntry.month && config.entries[i].day == 0 && config.entries[i].countryCode == "")
												{
													if (Array.isArray(config.entries[i].screen))
													{
														if (Array.isArray(cfgEntry.screen))
															config.entries[i].screen = config.entries[i].screen.concat(cfgEntry.screen);
														else
															config.entries[i].screen.push(cfgEntry.screen);
													}
													else
													{
														var tmpArray = [];
														tmpArray.push(config.entries[i].screen)
														if (Array.isArray(cfgEntry.screen))
															tmpArray = tmpArray.concat(cfgEntry.screen);
														else
															tmpArray.push(cfgEntry.screen);
														config.entries[i].screen = tmpArray;
													}
													foundEntry = true;
												}
											}
											if (!foundEntry)
												config.entries.push(cfgEntry);
										}
									}
								}
							}
						}
					}
				}
			}
		}
	}

	return config;
}

// Returns a default configuration entry
function getDefaultConfigEntry(pMonth, pDay, pCountryCode)
{
	var cfgEntry = {
		countryCode: "",
		month: 0,
		day: 0,
		screen: "" // Can be a single screen filename or an array, for a random screen
	};
	if (typeof(pMonth) === "number" && !isNaN(pMonth))
		cfgEntry.month = pMonth;
	if (typeof(pDay) === "number" && !isNaN(pDay))
		cfgEntry.day = pDay;
	if (typeof(pCountryCode) === "string")
		cfgEntry.countryCode = pCountryCode;
	return cfgEntry;
}

// Converts a day of weeek name or abbreviation (string) to a day of week number
// (0-6, where 0 is Sunday, or -1 for an invalid day)
function dayOfWeekNameToNum(pDayOfWeekStr)
{
	if (typeof(pDayOfWeekStr) !== "string")
		return -1;
	var dayStrLower = pDayOfWeekStr.toLowerCase();
	if (dayStrLower == "sun" || dayStrLower == "sunday")
		return 0;
	else if (dayStrLower == "mon" || dayStrLower == "monday")
		return 1;
	else if (dayStrLower == "tue" || dayStrLower == "tues" || dayStrLower == "tuesday")
		return 2;
	else if (dayStrLower == "wed" || dayStrLower == "weds" || dayStrLower == "wednesday")
		return 3;
	else if (dayStrLower == "thu" || dayStrLower == "thurs" || dayStrLower == "thursday")
		return 4;
	else if (dayStrLower == "fri" || dayStrLower == "friday")
		return 5;
	else if (dayStrLower == "sat" || dayStrLower == "saturday")
		return 6;
	else
		return -1;
}

// Figures out the day number of a month for an Nth weekday of the month.
// Returns -1 if it can't be determined.
//
// Parameters:
//  pMonthNum: The month number (1-31)
//  pDayOfWeekNum: The day of the week you're looking for (0-6 for Sunday-Saturday)
//  pOrdinalNum: The Nth day of the week to look for with the day of the week # given with the previous Parameters
//
// Return value: The resulting day of the month, or -1 if it can't be determined
function getNthWeekdayDayofMonthForCurrentYear(pMonthNum, pDayOfWeekNum, pOrdinalNum)
{
	var dayOfMonth = -1;
	// Make a date with the month number in this year at the first day
	var curYear = (new Date()).getFullYear();
	var tmpDate = new Date(curYear, pMonthNum-1);
	// Go through each day of the month looking for the given Nth day of the week
	var dayOfWeekCount = 0;
	var numDaysInMonth = new Date(curYear, pMonthNum-1, 0).getDate();
	for (var dayNum = 1; dayNum <= numDaysInMonth; ++dayNum)
	{
		tmpDate.setDate(dayNum);
		if (tmpDate.getDay() == pDayOfWeekNum)
			++dayOfWeekCount;
		if (dayOfWeekCount == pOrdinalNum)
		{
			dayOfMonth = tmpDate.getDate();
			break;
		}
	}
	return dayOfMonth;
}

// Parses a filename: If there's only 1, returns the filename as a string. If there
// are multiple separated by commas, returns an array of filenames (strings).
function parseScreenFilename(pFilename)
{
	if (typeof(pFilename) !== "string")
		return "";
	var filenames = pFilename.split(",");
	if (filenames.length == 0)
		return "";
	else if (filenames.length == 1)
		return filenames[0];
	else
		return filenames;
}

// Returns whether an item is a (non-empty) string or an array of non-empty strings
function itemIsStrOrArrayOfStrsAndNotBlank(pItem)
{
	if (typeof(pItem) === "string")
		return (pItem.length > 0);
	else if (Array.isArray(pItem))
	{
		if (pItem.length == 0)
			return false;
		else
		{
			var isAllNonEmptyStrs = true;
			for (var i = 0; i < pItem.length && isAllNonEmptyStrs; ++i)
				isAllNonEmptyStrs = (typeof(pItem[i]) === "string" && pItem[i].length > 0);
			return isAllNonEmptyStrs;
		}
	}
	else
		return false;
}