/*
Digital Distortion Animated Pause

This is a Synchronet script that acts as an animated pause prompt

Date       Author            Description
----       ------            -----------
2009-05-11 Eric Oulashin     Version 0.5
                             Initial development
2023-03-19 Eric Oulashin     Version 1.0
                             Now has input timeout and disconnects the user upon timeout.
*/

// Version 1.0

/*
To use this script, edit your text.dat (in the sbbs/ctrl directory) and change
line 563 (Pause) to:
"@EXEC:animatedPause@"
*/

"use strict";

if (typeof(require) === "function")
{
	require("sbbsdefs.js", "K_UPPER");
	require("text.js", "MsgSubj");
}
else
{
	load("sbbsdefs.js");
	load("text.js");
}

// Determine the location of this script (its startup directory).
// The code for figuring this out is a trick that was created by Deuce,
// suggested by Rob Swindell.  I've shortened the code a little.
var gStartupPath = '.';
try { throw dig.dist(dist); } catch(e) { gStartupPath = e.fileName; }
gStartupPath = backslash(gStartupPath.replace(/[\/\\][^\/\\]*$/,''));

// Block characters
var BLOCK1 = ""; // Dimmest block
var BLOCK2 = "";
var BLOCK3 = "";
var BLOCK4 = ""; // Brightest block
// Box-drawing/border characters: Single-line
var UPPER_LEFT_SINGLE = "";
var HORIZONTAL_SINGLE = "";
var UPPER_RIGHT_SINGLE = "";
var VERTICAL_SINGLE = "";
var LOWER_LEFT_SINGLE = "";
var LOWER_RIGHT_SINGLE = "";
var T_SINGLE = "";
var LEFT_T_SINGLE = "";
var RIGHT_T_SINGLE = "";
var BOTTOM_T_SINGLE = "";
var CROSS_SINGLE = "";
// Box-drawing/border characters: Double-line
var UPPER_LEFT_DOUBLE = "";
var HORIZONTAL_DOUBLE = "";
var UPPER_RIGHT_DOUBLE = "";
var VERTICAL_DOUBLE = "";
var LOWER_LEFT_DOUBLE = "";
var LOWER_RIGHT_DOUBLE = "";
var T_DOUBLE = "";
var LEFT_T_DOUBLE = "";
var RIGHT_T_DOUBLE = "";
var BOTTOM_T_DOUBLE = "";
var CROSS_DOUBLE = "";
// Box-drawing/border characters: Vertical single-line with horizontal double-line
var UPPER_LEFT_VSINGLE_HDOUBLE = "";
var UPPER_RIGHT_VSINGLE_HDOUBLE = "";
var LOWER_LEFT_VSINGLE_HDOUBLE = "";
var LOWER_RIGHT_VSINGLE_HDOUBLE = "";
// Other special characters
var DOT_CHAR = "";
var CHECK_CHAR = "";
var THIN_RECTANGLE_LEFT = "";
var THIN_RECTANGLE_RIGHT = "";

// Pause animation types
var PAUSE_CHARACTER_CYCLE = 1;      // Cycle through characters for a single character at the end
var PAUSE_CHARACTER_RANDOM = 2;     // Choose random characters from the character array for a single character at the end
var PAUSE_PROMPT_COLOR_CYCLE = 3;   // Cycle through colors of the pause prompt characters
var PAUSE_PROMPT_RANDOM_COLORS = 4; // Randomize the colors of the pause prompt characters
var PAUSE_PROMPT_TRAIN_LEFT = 5;    // "Train" style movement with characters moving left
var PAUSE_PROMPT_TRAIN_RIGHT = 6;   // "Train" style movement with characters moving right
var PAUSE_PROMPT_ROTATE_LEFT = 7;   // Rotating the pause text left
var PAUSE_PROMPT_ROTATE_RIGHT = 8;  // Rotating the pause text right



////////////////////////////
// Script execution

// Read the configuration file and store the configuration settings in a
// configuration object.  ReadConfigFile() function will set defaults before
// reading the configuration file, so the configuration defaults don't need to
// be set here.
var gCfgObj = ReadConfigFile();

// Display the animated pause prompt if the user has their animated pause
// prompt setting enabled and the user's terminal supports ANSI.
var userInput = "";
if (userSpinPauseEnabled() && console.term_supports(USER_ANSI))
{
	var initialTime = system.timer;
	switch (gCfgObj.animationType)
	{
		case PAUSE_CHARACTER_CYCLE:
		default:
			// Display the pause text
			console.print("\x01n" + gCfgObj.pauseText);
			// Animate a character after the pause text
			var curPos = console.getxy();
			var cursorCharIdx = 0;
			var pauseTextLen = strip_ctrl(gCfgObj.pauseText).length;
			// User input loop with changing cursor
			while (userInput === "" && (system.timer - initialTime) * 1000 <= gCfgObj.inputTimeoutMS)
			{
				// Display a character - Cycle through the animated cursor characters
				console.gotoxy(curPos);
				console.print("\x01n" + gCfgObj.cursorSpinColor + gCfgObj.animatedCursorChars[cursorCharIdx++]);
				if (cursorCharIdx >= gCfgObj.animatedCursorChars.length)
					cursorCharIdx = 0;
				// Get a key from the user with a timeout
				userInput = console.inkey(K_NOSPIN|K_NOCRLF|K_NOECHO, gCfgObj.animationTimeoutMS);
			}
			break;
		case PAUSE_CHARACTER_RANDOM:
			// Display the pause text
			console.print("\x01n" + gCfgObj.pauseText);
			// Animate a character after the pause text
			var curPos = console.getxy();
			var cursorCharIdx = 0;
			var pauseTextLen = strip_ctrl(gCfgObj.pauseText).length;
			// User input loop with changing cursor
			while (userInput === "" && (system.timer - initialTime) * 1000 <= gCfgObj.inputTimeoutMS)
			{
				// Display a random character from the animated cursor characters
				cursorCharIdx = Math.floor(Math.random() * (gCfgObj.animatedCursorChars.length-1));
				console.gotoxy(curPos);
				console.print("\x01n" + gCfgObj.cursorSpinColor + gCfgObj.animatedCursorChars[cursorCharIdx]);
				// Get a key from the user with a timeout
				userInput = console.inkey(K_NOSPIN|K_NOCRLF|K_NOECHO, gCfgObj.animationTimeoutMS);
			}
			break;
		case PAUSE_PROMPT_COLOR_CYCLE:
			var pauseTextNoAttrs = strip_ctrl(gCfgObj.pauseText);
			var curPos = console.getxy();
			var colorIdx = 0;
			// User input loop with prompt colors cycling through colorCodes
			while (userInput === "" && (system.timer - initialTime) * 1000 <= gCfgObj.inputTimeoutMS)
			{
				console.gotoxy(curPos);
				if (pauseTextNoAttrs.length == gCfgObj.colorCodes.length)
				{
					if (colorIdx >= gCfgObj.colorCodes.length)
						colorIdx = 0;
					++colorIdx;
				}
				for (var pauseTextIdx = 0; pauseTextIdx < pauseTextNoAttrs.length; ++pauseTextIdx)
				{
					console.print("\x01n" + gCfgObj.colorCodes[colorIdx++] + pauseTextNoAttrs[pauseTextIdx]);
					if (colorIdx >= gCfgObj.colorCodes.length)
						colorIdx = 0;
				}
				// Get a key from the user with a timeout
				console.attributes = "N";
				userInput = console.inkey(K_NOSPIN|K_NOCRLF|K_NOECHO, gCfgObj.animationTimeoutMS);
			}
			break;
		case PAUSE_PROMPT_RANDOM_COLORS:
			var pauseTextNoAttrs = strip_ctrl(gCfgObj.pauseText);
			var curPos = console.getxy();
			var colorIdx = 0;
			// User input loop with prompt colors chosen randomly from colorCodes
			while (userInput === "" && (system.timer - initialTime) * 1000 <= gCfgObj.inputTimeoutMS)
			{
				console.gotoxy(curPos); // Error: can't convert "\x01c[ Press a key ] " to an integer
				for (var pauseTextIdx = 0; pauseTextIdx < pauseTextNoAttrs.length; ++pauseTextIdx)
				{
					colorIdx = Math.floor(Math.random() * (gCfgObj.colorCodes.length-1));
					console.print("\x01n" + gCfgObj.colorCodes[colorIdx] + pauseTextNoAttrs[pauseTextIdx]);
				}
				// Get a key from the user with a timeout
				console.attributes = "N";
				userInput = console.inkey(K_NOSPIN|K_NOCRLF|K_NOECHO, gCfgObj.animationTimeoutMS);
			}
			break;
		case PAUSE_PROMPT_TRAIN_LEFT:
			var pauseTextNoAttrs = strip_ctrl(gCfgObj.pauseText);
			var pauseTextLen = pauseTextNoAttrs.length;
			var curPos = console.getxy();
			// Blank out the width of the pause text where the cursor currently is
			console.attributes = "N";
			for (var i = 0; i < pauseTextLen; ++i)
				console.print(" ");
			// User input loop with the prompt text moving to the left
			var textLen = pauseTextLen;
			while (userInput === "" && (system.timer - initialTime) * 1000 <= gCfgObj.inputTimeoutMS)
			{
				console.gotoxy(curPos);
				var pauseText = shortenStrWithAttrCodes(gCfgObj.pauseText, textLen, false);
				console.print(pauseText);
				// Clear the space after the pause text
				var numSpaces = pauseTextLen - textLen;
				for (var i = 0; i < numSpaces; ++i)
					console.print(" ");
				// Update the text length for the next pass through the loop
				--textLen;
				if (textLen < 0)
					textLen = pauseTextLen;
				// Get a key from the user with a timeout
				console.attributes = "N";
				userInput = console.inkey(K_NOSPIN|K_NOCRLF|K_NOECHO, gCfgObj.animationTimeoutMS);
			}
			break;
		case PAUSE_PROMPT_TRAIN_RIGHT:
			var pauseTextNoAttrs = strip_ctrl(gCfgObj.pauseText);
			var pauseTextLen = pauseTextNoAttrs.length;
			var curPos = console.getxy();
			// Blank out the width of the pause text where the cursor currently is
			console.attributes = "N";
			for (var i = 0; i < pauseTextLen; ++i)
				console.print(" ");
			// User input loop with the prompt text moving to the left
			var textLen = pauseTextLen;
			while (userInput === "" && (system.timer - initialTime) * 1000 <= gCfgObj.inputTimeoutMS)
			{
				//function shortenStrWithAttrCodes(pStr, pNewLength, pFromLeft)
				console.gotoxy(curPos);
				var pauseText = shortenStrWithAttrCodes(gCfgObj.pauseText, textLen, true);
				// Clear the space before the pause text
				var numSpaces = pauseTextLen - textLen;
				for (var i = 0; i < numSpaces; ++i)
					console.print(" ");
				// Print the pause text
				console.print(pauseText);
				// Update the text length for the next pass through the loop
				--textLen;
				if (textLen < 0)
					textLen = pauseTextLen;
				// Get a key from the user with a timeout
				console.attributes = "N";
				userInput = console.inkey(K_NOSPIN|K_NOCRLF|K_NOECHO, gCfgObj.animationTimeoutMS);
			}
			break;
		case PAUSE_PROMPT_ROTATE_LEFT:
			var curPos = console.getxy();
			// User input loop with the prompt text rotating left
			var pauseText = gCfgObj.pauseText;
			var stringRotateAmount = 1;
			while (userInput === "" && (system.timer - initialTime) * 1000 <= gCfgObj.inputTimeoutMS)
			{
				console.gotoxy(curPos);
				console.print(pauseText);
				// Get a key from the user with a timeout
				console.attributes = "N";
				userInput = console.inkey(K_NOSPIN|K_NOCRLF|K_NOECHO, gCfgObj.animationTimeoutMS);
				// Rotate the pause text
				pauseText = rotateStr(gCfgObj.pauseText, stringRotateAmount++, true);
			}
			break;
		case PAUSE_PROMPT_ROTATE_RIGHT:
			var curPos = console.getxy();
			// User input loop with the prompt text rotating left
			var pauseText = gCfgObj.pauseText;
			var stringRotateAmount = 1;
			while (userInput === "" && (system.timer - initialTime) * 1000 <= gCfgObj.inputTimeoutMS)
			{
				console.gotoxy(curPos);
				console.print(pauseText);
				// Get a key from the user with a timeout
				console.attributes = "N";
				userInput = console.inkey(K_NOSPIN|K_NOCRLF|K_NOECHO, gCfgObj.animationTimeoutMS);
				// Rotate the pause text
				pauseText = rotateStr(gCfgObj.pauseText, stringRotateAmount++, false);
			}
			break;
	}
}
else
{
	console.print(gCfgObj.pauseText);
	userInput = console.inkey(K_NOSPIN|K_NOCRLF|K_NOECHO, gCfgObj.inputTimeoutMS);
	//console.pause();
}
// Put another return into the input buffer - It seems this is needed, because
// after getting input from the user for pausing, Synchronet would pause again
// and wait for another keypress:
console.ungetstr("\r");

// If the user's input is null or an empty string, they didn't press a key; disconnect
// the user due to timeout (if they're not the sysop)
if (userInput == null || (typeof(userInput) === "string" && userInput.length == 0))
{
	if (!user.is_sysop)
	{
		// Disconnect
		console.putmsg(bbs.text(CallBackWhenYoureThere));
		console.crlf();
		bbs.hangup();
		//bbs.logoff(false);
	}
}
else if (typeof(userInput) === "string")
{
	var userInputUpper = userInput.toUpperCase();
	// Q or N should set console.aborted to true
	console.aborted = (userInputUpper === "Q" || userInputUpper === "N");
}
else
	console.aborted = false;

// Put a key into the buffer to avoid an extra pause at the end.
//console.ungetstr("\x0d"); // Enter (AKA Ctrl-M)
//console.crlf();

// Set the console line counter to 0 to avoid another pause when this script
// exits
console.line_counter = 0;
console.current_column = 0;



////////////////////////////////////////
// Functions

// Returns whether or not the user's spinning pause prompt setting is enabled.
function userSpinPauseEnabled()
{
	//return ((user.settings & USER_SPIN) == USER_SPIN);
	return ((user.settings & USER_NOPAUSESPIN) == 0);
}

// Removes multiple, leading, and/or trailing spaces
// The search & replace regular expressions used in this
// function came from the following URL:
// http://qodo.co.uk/blog/javascript-trim-leading-and-trailing-spaces
//
// Parameters:
//  pString: The string to trim
//  pLeading: Whether or not to trim leading spaces (optional, defaults to true)
//  pMultiple: Whether or not to trim multiple spaces (optional, defaults to true)
//  pTrailing: Whether or not to trim trailing spaces (optional, defaults to true)
//
// Return value: The trimmed string
function trimSpaces(pString, pLeading, pMultiple, pTrailing)
{
   // Make sure pString is a string.
   if (typeof(pString) == "string")
   {
      var leading = true;
      var multiple = true;
      var trailing = true;
      if(typeof(pLeading) != "undefined")
         leading = pLeading;
      if(typeof(pMultiple) != "undefined")
         multiple = pMultiple;
      if(typeof(pTrailing) != "undefined")
         trailing = pTrailing;

      // To remove both leading & trailing spaces:
      //pString = pString.replace(/(^\s*)|(\s*$)/gi,"");

      if (leading)
         pString = pString.replace(/(^\s*)/gi,"");
      if (multiple)
         pString = pString.replace(/[ ]{2,}/gi," ");
      if (trailing)
         pString = pString.replace(/(\s*$)/gi,"");
   }

   return pString;
}

// Reads the configuration file.  Returns an object containing the configuration
// settings.
function ReadConfigFile()
{
	var cfgObj = new Object(); // Configuration object

	// Default settings
	// pauseText is the text to display for the pause prompt, before the
	// spinning/animated character
	cfgObj.pauseText = "[Press a key] ";
	// animationType is the type of animation to use for the pause prompt
	cfgObj.animationType = PAUSE_CHARACTER_CYCLE;
	//cfgObj.animationType = PAUSE_PROMPT_COLOR_CYCLE;
	//cfgObj.animationType = PAUSE_PROMPT_RANDOM_COLORS;
	// cursorSpinColor is the color to use for the spinning/animated character
	cfgObj.cursorSpinColor = "\x01g";
	// animationTimeoutMS is the animation wait timeout (in milliseconds).  Controls the spinning
	// character speed.
	cfgObj.animationTimeoutMS = 250;
	// inputTimeoutMS is the timeout (in milliseconds) for input when waiting for the
	// user to press a key
	cfgObj.inputTimeoutMS = 30000;
	// animatedCursorChars is an array of characters to cycle through for the animated/spinning
	// prompt.
	//cfgObj.animatedCursorChars = [ BLOCK1, BLOCK2, BLOCK3, BLOCK4 ];
	//cfgObj.animatedCursorChars = [ UPPER_LEFT_SINGLE, HORIZONTAL_SINGLE, UPPER_RIGHT_SINGLE, VERTICAL_SINGLE, LOWER_LEFT_SINGLE, LOWER_RIGHT_SINGLE, T_SINGLE, RIGHT_T_SINGLE, BOTTOM_T_SINGLE, CROSS_SINGLE ];
	cfgObj.animatedCursorChars = [ BLOCK1, BLOCK2, BLOCK3, BLOCK4, UPPER_LEFT_SINGLE, HORIZONTAL_SINGLE, UPPER_RIGHT_SINGLE, VERTICAL_SINGLE, LOWER_LEFT_SINGLE, LOWER_RIGHT_SINGLE, T_SINGLE, RIGHT_T_SINGLE, BOTTOM_T_SINGLE, CROSS_SINGLE, "!", "?", "*", "<", "^", ">" ];

	/*
		Foreground	Background
		----------	----------
	Black       K		    0
	Red         R		    1
	Green       G		    2
	Yellow      Y		    3
	Blue        B		    4
	Magenta     M		    5
	Cyan        C		    6
	White       W		    7

		Attribute	Description
		---------	-----------
	High        H		High Intensity
	*/
	// colorCodes is an array of colors to cycle through for
	// PAUSE_PROMPT_COLOR_CYCLE and PAUSE_PROMPT_RANDOM_COLORS.
	cfgObj.colorCodes = [ "\x01n\x01r", "\x01n\x01g", "\x01n\x01y", "\x01n\x01b", "\x01n\x01m", "\x01n\x01c", "\x01n\x01w",
	                      "\x01n\x01h\x01k", "\x01n\x01h\x01r", "\x01n\x01h\x01g", "\x01n\x01h\x01y", "\x01n\x01h\x01b",
	                      "\x01n\x01h\x01m", "\x01n\x01h\x01c", "\x01n\x01h\x01w" ];

	// Open the configuration file
	var cfgFileName = genFullPathCfgFilename("animatedPause.cfg", gStartupPath);
	var cfgFile = new File(cfgFileName);
	if (cfgFile.open("r"))
	{
		var fileLine = null;     // A line read from the file
		var equalsPos = 0;       // Position of a = in the line
		var commentPos = 0;      // Position of the start of a comment
		var setting = null;      // A setting name (string)
		var settingUpper = null; // Upper-case setting name
		var valueTrimmed = null;  // A value for a setting (string), with spaces trimmed
		var valueLiteral = null; // The value as it is in the config file, no processing
		var valueUpper = null;   // Upper-cased value
		while (!cfgFile.eof)
		{
			// Read the next line from the config file.
			fileLine = cfgFile.readln(2048);

			// fileLine should be a string, but I've seen some cases
			// where for some reason it isn't.  If it's not a string,
			// then continue onto the next line.
			if (typeof(fileLine) != "string")
				continue;

			// If the line starts with with a semicolon (the comment
			// character) or is blank, then skip it.
			if ((fileLine.substr(0, 1) == ";") || (fileLine.length == 0))
				continue;

			// If the line has a semicolon anywhere in it, then remove
			// everything from the semicolon onward.
			commentPos = fileLine.indexOf(";");
			if (commentPos > -1)
				fileLine = fileLine.substr(0, commentPos);

			// Look for an equals sign, and if found, separate the line
			// into the setting name (before the =) and the value (after the
			// equals sign).
			equalsPos = fileLine.indexOf("=");
			if (equalsPos > 0)
			{
				// Read the setting & value, and trim leading & trailing spaces.
				setting = trimSpaces(fileLine.substr(0, equalsPos), true, false, true);
				settingUpper = setting.toUpperCase();
				valueLiteral = fileLine.substr(equalsPos+1);
				valueTrimmed = trimSpaces(valueLiteral, true, false, true);
				valueUpper = valueLiteral.toUpperCase();

				if (settingUpper == "ANIMATIONTYPE")
				{
					if (valueUpper === "PAUSE_CHARACTER_CYCLE")
						cfgObj.animationType = PAUSE_CHARACTER_CYCLE;
					else if (valueUpper === "PAUSE_CHARACTER_RANDOM")
						cfgObj.animationType = PAUSE_CHARACTER_RANDOM;
					else if (valueUpper === "PAUSE_PROMPT_COLOR_CYCLE")
						cfgObj.animationType = PAUSE_PROMPT_COLOR_CYCLE;
					else if (valueUpper === "PAUSE_PROMPT_RANDOM_COLORS")
						cfgObj.animationType = PAUSE_PROMPT_RANDOM_COLORS;
					else if (valueUpper === "PAUSE_PROMPT_TRAIN_LEFT")
						cfgObj.animationType = PAUSE_PROMPT_TRAIN_LEFT;
					else if (valueUpper === "PAUSE_PROMPT_TRAIN_RIGHT")
						cfgObj.animationType = PAUSE_PROMPT_TRAIN_RIGHT;
					else if (valueUpper === "PAUSE_PROMPT_ROTATE_LEFT")
						cfgObj.animationType = PAUSE_PROMPT_ROTATE_LEFT;
					else if (valueUpper === "PAUSE_PROMPT_ROTATE_RIGHT")
						cfgObj.animationType = PAUSE_PROMPT_ROTATE_RIGHT;
					else if (valueUpper === "RANDOM")
						cfgObj.animationType = randomInt(1, 8);
				}
				else if (settingUpper == "PAUSETEXT")
				{
					if (valueLiteral.length > 0)
						cfgObj.pauseText = valueLiteral;
				}
				else if (settingUpper == "CURSORSPINCOLOR")
					cfgObj.cursorSpinColor = valueTrimmed;
				else if (settingUpper == "ANIMATIONTIMEOUTMS")
				{
					// Only set this if the value is all digits and positive.
					var valInt = parseInt(valueLiteral);
					if (!isNaN(valInt) && valInt > 0)
						cfgObj.animationTimeoutMS = valInt;
				}
				else if (settingUpper == "INPUTTIMEOUTMS")
				{
					// Only set this if the value is all digits and positive.
					var valInt = parseInt(valueLiteral);
					if (!isNaN(valInt) && valInt > 0)
						cfgObj.inputTimeoutMS = valInt;
				}
				else if (settingUpper == "ANIMATEDCURSORCHARS")
					cfgObj.animatedCursorChars = valueLiteral.split(",");
				else if (settingUpper == "COLORCODES")
					cfgObj.colorCodes = valueLiteral.split(",");
			}
		}

		cfgFile.close();

		// Allow \x01 for the control character in color codes
		cfgObj.pauseText = cfgObj.pauseText.replace(/\\x01/g, "\x01").replace(/\\1/g, "\x01");
		cfgObj.cursorSpinColor = cfgObj.cursorSpinColor.replace(/\\x01/g, "\x01").replace(/\\1/g, "\x01");
		for (var i = 0; i < cfgObj.animatedCursorChars.length; ++i)
			cfgObj.animatedCursorChars[i] = cfgObj.animatedCursorChars[i].replace(/\\x01/g, "\x01").replace(/\\1/g, "\x01");
		for (var i = 0; i < cfgObj.colorCodes.length; ++i)
			cfgObj.colorCodes[i] = cfgObj.colorCodes[i].replace(/\\x01/g, "\x01").replace(/\\1/g, "\x01");
	}

	return cfgObj;
}

// For configuration files, this function returns a fully-pathed filename.
// This function first checks to see if the file exists in the sbbs/mods
// directory, then the sbbs/ctrl directory, and if the file is not found there,
// this function defaults to the given default path.
//
// Parameters:
//  pFilename: The name of the file to look for
//  pDefaultPath: The default directory (must have a trailing separator character)
function genFullPathCfgFilename(pFilename, pDefaultPath)
{
	var fullyPathedFilename = system.mods_dir + pFilename;
	if (!file_exists(fullyPathedFilename))
		fullyPathedFilename = system.ctrl_dir + pFilename;
	if (!file_exists(fullyPathedFilename))
	{
		if (typeof(pDefaultPath) == "string")
		{
			// Make sure the default path has a trailing path separator
			var defaultPath = backslash(pDefaultPath);
			fullyPathedFilename = defaultPath + pFilename;
		}
		else
			fullyPathedFilename = pFilename;
	}
	return fullyPathedFilename;
}

// Shortens a string, accounting for control/attribute codes.  Returns a new
// (shortened) copy of the string.
//
// Parameters:
//  pStr: The string to shorten
//  pNewLength: The new (shorter) length of the string
//  pFromLeft: Optional boolean - Whether to start from the left (default) or
//             from the right.
//
// Return value: The shortened version of the string
function shortenStrWithAttrCodes(pStr, pNewLength, pFromLeft)
{
	if (typeof(pStr) != "string")
		return "";
	if (typeof(pNewLength) != "number")
		return pStr;
	if (pNewLength >= console.strlen(pStr))
		return pStr;

	var fromLeft = (typeof(pFromLeft) == "boolean" ? pFromLeft : true);
	var strCopy = "";
	var tmpStr = "";
	var strIdx = 0;
	var lengthGood = true;
	if (fromLeft)
	{
		while (lengthGood && (strIdx < pStr.length))
		{
			tmpStr = strCopy + pStr.charAt(strIdx++);
			if (console.strlen(tmpStr) <= pNewLength)
				strCopy = tmpStr;
			else
				lengthGood = false;
		}
	}
	else
	{
		strIdx = pStr.length - 1;
		while (lengthGood && (strIdx >= 0))
		{
			tmpStr = pStr.charAt(strIdx--) + strCopy;
			if (console.strlen(tmpStr) <= pNewLength)
				strCopy = tmpStr;
			else
				lengthGood = false;
		}
	}
	return strCopy;
}

// Rotates a string
//
// Parameters:
//  pStr: The string to rotate
//  pRotateAmount: The number of characters to rotate the string by
//  pLeft: Optional boolean - Whether or not to rotate left (default).  If this
//         is false, the string will be rotated right.
//
// Return value: The rotated string
function rotateStr(pStr, pRotateAmount, pLeft)
{
	if (pStr.length < 2)
		return pStr;

	// The basic idea for this was found here:
	// http://www.w3resource.com/javascript-exercises/javascript-basic-exercise-5.php
	var str = pStr;
	var goLeft = (typeof(pLeft) == "boolean" ? pLeft : true);
	if (goLeft)
	{
		// If the beginning of the string contains a Synchronet attribute
		// control character, then remove it and the next character to
		// prevent a weird character appearing on the screen and to leave
		// the string length correct.
		//if ((pRotateAmount == 1) && (str.charAt(0) == "\x01"))
		//	str = str.substr(2);
		for (var i = 0; i < pRotateAmount; ++i)
			str = str.substring(1, str.length) + str.charAt(0);
	}
	else
	{
		for (var i = 0; i < pRotateAmount; ++i)
			str = str[str.length - 1] + str.substring(0, str.length - 1);
	}

	// If the string ends with a Synchronet color code control character(\x01),
	// then remove it (to avoid a weird character appearing on the screen),
	// and also remove the first character since it would go along with the
	// Synchronet control character.
	if (str.charAt(str.length-1) == "")
		str = str.substring(0, str.length-1).substring(1);

	return str;
}

// Returns a random number between a minimum & maximum value (inclusive).
//
// Parameters:
//  pMinValue: The minimum value
//  pMaxValue: The maximum value
function randomInt(pMinValue, pMaxValue)
{
	return Math.floor((Math.random() * pMaxValue) + pMinValue);
}